/**
 * \file keyman.h
 *
 * \brief Libkeyman external interface functions
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#ifndef __LIBKEYMAN_H_
#define __LIBKEYMAN_H_

#include <stdbool.h>
#include <sdc.h>
#include <sdc_perm.h>

/**
 * \defgroup libKeyman libKeyman
 */

/**
 * \ingroup libKeyman
 * \defgroup libKeyman_API libKeyman API
 *
 * The SDC daemon can(but does not need to) be used in multiple architectures.
 * In case the SDC daemon is used libSDC requests to install or remove keys are forwarded to the SDC daemon.
 * The SDC daemon will use libKeyman in order to fulfill these requests.
 * Furthermore the SDC daemon can be used to import key containers.
 *
 * This sections describes all the functions and types used by the SDC daemon to perform its job.
 */

/**
 * \ingroup libKeyman_API
 * \brief Used to specify that this function is part of the "normal" API of libKeyman
 *
 * These function will be visible to applications including libkeyman library */
#define KEYMAN_API __attribute__((__visibility__("default")))

/**
 * \ingroup libKeyman_API
 * \brief LibKeyman Context
 *
 * The SDC daemon will call \ref keyman_ctx_alloc to create the
 * context before calling any function like \ref keyman_load_persistent_keystore.
 * This structure is used to store the architecture specific context (e.g. file descriptor of kernel interface)
 */
typedef struct libkeyman_ctx libkeyman_ctx_t;

/**
 * \ingroup libKeyman_API
 * \brief specify that a operation applies to the complete keystore or only volatile or persistent
 *
 * Used to specify if \ref keyman_flush_keystore and \ref keyman_lock_unlock_keystore will affect
 * the complete keystore or only volatile or persistent keys of the keystore.
 *\todo Currently ignored
 */
typedef enum libkeyman_keystore_cont_t {
    LIBKEYMAN_KEYSTORE_ALL, /**< complete keystore */
    LIBKEYMAN_KEYSTORE_PERSISTENT, /**< persistent keys of keystore only */
    LIBKEYMAN_KEYSTORE_NON_PERSISTENT /**< volatile keys of keystore only */
} libkeyman_keystore_cont_t;


/**
 * \ingroup libKeyman_API
 * \brief struct containing settings common to all key installation functions
 */
struct libkeyman_key_spec {
    sdc_key_fmt_t fmt;
    sdc_key_len_t len;
    const sdc_permissions_t *permissions;
};
/**
 * \ingroup libKeyman_API
 * \brief type containing settings common to all key installation functions
 */
typedef struct libkeyman_key_spec libkeyman_key_spec_t;


/**
 * \ingroup libKeyman_API
 * \brief Verify the kernel version of the libKeyman kernel interface.
 *
 * No initialization of the libkeyman context is required
 *
 * \param[out] version_info_str optional pointer to return detailed error information (needs to be freed externally)
 *
 * \return \ref SDC_OK on success, else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_kernel_version_verify(char **version_info_str);

/**
 * \ingroup libKeyman_API
 *
 * \brief Allocate and initialize the context before first usage
 *
 * Initially the key-store should be locked
 *
 * \param[out] ctx pointer to store the internally allocated context
 *
 * \return \ref SDC_OK on success, else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_ctx_alloc(libkeyman_ctx_t **ctx);

/**
 * \ingroup libKeyman_API
 * \brief Deinitialize and free the context
 *
 * This function will also free memory allocated by the libkeyman context
 *
 * \param[in] ctx pointer to libkeyman context
 *
 * \return \ref SDC_OK on success, else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_ctx_free(libkeyman_ctx_t *ctx);

/**
 * \ingroup libKeyman_API
 * \brief Restore the content of the key-store (persistent keystorage)
 *
 * \param[in] ctx pointer to libkeyman context
 * \param[in] fd file-descriptor of the opened (read) key-store file which shall be restored
 * \param[out] loaded_keys optional pointer to return the number of loaded keys
 * \param[out] existing_keys optional pointer to return the number of keys which have not been loaded as the key is already present
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_KEY_STORAGE_ACCESS_FAILED - access failed
 * \return \ref SDC_KEY_STORAGE_CORRUPT - key-store content is corrupt
 * \return \ref SDC_KEY_TAMPERED - a key in the key-store has been tampered
 * \return else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_load_persistent_keystore(libkeyman_ctx_t *ctx, int fd,
                                                       size_t *loaded_keys, size_t *existing_keys);

/**
 * \ingroup libKeyman_API
 * \brief Store the persistent keystorage
 *
 * Key store file data sync needs to be handled inside arch specific part.
 *
 * \param[in] ctx pointer to libkeyman context
 * \param[in] fd file-descriptor of the opened (write) key-store file which shall be written
 * \param[out] stored_keys optional pointer to return the number of stored keys
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_KEY_STORAGE_ACCESS_FAILED - access failed
 * \return else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_store_persistent_keystore(libkeyman_ctx_t *ctx, int fd, size_t *stored_keys);

/**
 * \ingroup libKeyman_API
 * \brief Create an empty persistent keystorage
 *
 * \param[in] ctx pointer to libkeyman context
 * \param[in] fd file-descriptor of the opened (write) key-store file which shall be written
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_KEY_STORAGE_ACCESS_FAILED - access failed
 * \return else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_empty_persistent_keystore(libkeyman_ctx_t *ctx, int fd);

/**
 * \ingroup libKeyman_API
 * \brief Flush the content of the key-storage
 *
 * It is possible to call this operation while key-store is locked
 *
 * Currently \p keystore_content is limited to \ref LIBKEYMAN_KEYSTORE_ALL
 *
 * \param[in] ctx pointer to libkeyman context
 * \param[in] keystore_content specification of the keys to be flushed
 * \param[in] force force operation
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_KEY_NOT_READY a key is still in use (only if force==false)
 * \return else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_flush_keystore(libkeyman_ctx_t *ctx,
                                             libkeyman_keystore_cont_t keystore_content, bool force);

/**
 * \ingroup libKeyman_API
 * \brief Lock/Unlock the content of the key-storage
 *
 * Currently \p keystore_content is limited to \ref LIBKEYMAN_KEYSTORE_ALL
 *
 * While locked setting a corresponding key in a session will fail with
 * SDC_KEY_LOCKED
 * Locking will only prevent usage of key - install and remove is still possible
 * It is expected that the system is initially locked
 *
 * \param[in] ctx pointer to libkeyman context
 * \param[in] keystore_content specification of the keys to be flushed
 * \param[in] lock set lock if true - else release lock
 *
 * \return \ref SDC_OK on success
 * \return else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_lock_unlock_keystore(libkeyman_ctx_t *ctx,
                                                   libkeyman_keystore_cont_t keystore_content, bool lock);

/**
 * \ingroup libKeyman_API
 * \brief Install a key with an explicit KID and a known secret
 *
 * \param[in] ctx pointer to libkeyman context
 * \param[in] key_id id of the key
 * \param[in] key_spec struct containing format, length and permission settings
 * \param[in] persistent key shall be persistent
 * \param[in] activate_immediately if true key will be activated immediately
 * \param[in] key_data_enc externally defined encoding of \p key_data
 * \param[in] key_data key secret
 * \param[in] key_data_len length of the secret
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_KID_EXISTS key id is already used
 * \return \ref SDC_KEYLEN_INVALID invalid key length
 * \return else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_generate_plain_storage_key_explicit_kid(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately,
    sdc_keysecret_enc_t key_data_enc,
    const uint8_t *key_data,
    size_t key_data_len);

/**
 * \ingroup libKeyman_API
 * \brief Install a key with an automatically determined KID and a known secret
 *
 * The KID will be a random value in the range of \p key_id_first and \p key_id_last.
 * The KID will be returned using \p result_key_id
 *
 * \param[in] ctx pointer to libkeyman context
 * \param[in] key_id_first lower border of the KID range
 * \param[in] key_id_last upper border of the KID range
 * \param[out] result_key_id KID of new key
 * \param[in] key_spec struct containing format, length and permission settings
 * \param[in] persistent key shall be persistent
 * \param[in] activate_immediately if true key will be activated immediately
 * \param[in] key_data_enc externally defined encoding of \p key_data
 * \param[in] key_data key secret
 * \param[in] key_data_len length of the secret
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_KID_EXISTS all KIDs in the range already used
 * \return \ref SDC_KEYLEN_INVALID invalid key length
 * \return else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_generate_plain_storage_key_automatic_kid(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id_first,
    sdc_key_id_t key_id_last,
    sdc_key_id_t *result_key_id,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately,
    sdc_keysecret_enc_t key_data_enc,
    const uint8_t *key_data,
    size_t key_data_len);

/**
 * \ingroup libKeyman_API
 * \brief Install a key with an explicit KID and an unknown secret
 *
 * \param[in] ctx pointer to libkeyman context
 * \param[in] key_id id of the key
 * \param[in] key_spec struct containing format, length and permission settings
 * \param[in] persistent key shall be persistent
 * \param[in] activate_immediately if true key will be activated immediately
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_KID_EXISTS key id is already used
 * \return \ref SDC_KEYLEN_INVALID invalid key length
 * \return else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_generate_random_storage_key_explicit_kid(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately);

/**
 * \ingroup libKeyman_API
 * \brief Install a key with an automatically determined KID and an unknown secret
 *
 * The KID will be a random value in the range of \p key_id_first and \p key_id_last.
 * The KID will be returned using \p result_key_id
 *
 * \param[in] ctx pointer to libkeyman context
 * \param[in] key_id_first lower border of the KID range
 * \param[in] key_id_last upper border of the KID range
 * \param[out] result_key_id KID of new key
 * \param[in] key_spec struct containing format, length and permission settings
 * \param[in] persistent key shall be persistent
 * \param[in] activate_immediately if true key will be activated immediately
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_KID_EXISTS all KIDs in the range already used
 * \return \ref SDC_KEYLEN_INVALID invalid key length
 * \return else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_generate_random_storage_key_automatic_kid(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id_first,
    sdc_key_id_t key_id_last,
    sdc_key_id_t *result_key_id,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately);

/**
 * \ingroup libKeyman_API
 * \brief Import a key (from arch specific encrypted key data) with an explicit KID
 *
 * Import a wrapped key into the keystore consists out
 * of two steps.
 * 1st: sdc_arch_wrapped_storage_key_to_daemon_data is
 * used to generate architecture specific daemon_data.
 * 2nd(this function): Installing the key from architecture
 * specific daemon data(key_data) to the keystore.
 *
 * The other parameters are simply forwarded.
 *
 * \param[in] ctx pointer to libkeyman context
 * \param[in] key_id id of the key
 * \param[in] key_spec struct containing format, length and permission settings
 * \param[in] persistent key shall be persistent
 * \param[in] activate_immediately if true key will be activated immediately
 * \param[in] key_data_enc externally defined encoding of \p key_data
 * \param[in] key_data arch specific encrypted key data
 * \param[in] key_data_len length of \p key_data
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_KID_EXISTS key id is already used
 * \return \ref SDC_KEYLEN_INVALID invalid key length
 * \return else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_import_storage_key_explicit_kid(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately,
    sdc_keysecret_enc_t key_data_enc,
    const uint8_t *key_data,
    size_t key_data_len);

/**
 * \ingroup libKeyman_API
 * \brief Import a key (from arch specific encrypted key data) with an automatically determined KID
 *
 * Import a wrapped key into the keystore consists out
 * of two steps.
 * 1st: sdc_arch_wrapped_storage_key_to_daemon_data is
 * used to generate architecture specific daemon_data.
 * 2nd(this function): Installing the key from architecture
 * specific daemon data(key_data) to the keystore.
 *
 * The other parameters are simply forwarded.
 *
 * The KID will be a random value in the range of \p key_id_first and \p key_id_last.
 * The KID will be returned using \p result_key_id
 *
 * \param[in] ctx pointer to libkeyman context
 * \param[in] key_id_first lower border of the KID range
 * \param[in] key_id_last upper border of the KID range
 * \param[out] result_key_id KID of new key
 * \param[in] key_spec struct containing format, length and permission settings
 * \param[in] persistent key shall be persistent
 * \param[in] activate_immediately if true key will be activated immediately
 * \param[in] key_data_enc externally defined encoding of \p key_data
 * \param[in] key_data arch specific encrypted key data
 * \param[in] key_data_len length of \p key_data
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_KID_EXISTS all KIDs in the range already used
 * \return \ref SDC_KEYLEN_INVALID invalid key length
 * \return else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_import_storage_key_automatic_kid(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id_first,
    sdc_key_id_t key_id_last,
    sdc_key_id_t *result_key_id,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately,
    sdc_keysecret_enc_t key_data_enc,
    const uint8_t *key_data,
    size_t key_data_len);

/**
 * \ingroup libKeyman_API
 * \brief Special function to import an product key (insecure way - plain secret)
 *
 * These keys are persistent by definition.
 *
 * \param[in] ctx pointer to libkeyman context
 * \param[in] key_id id of the key
 * \param[in] key_spec struct containing format, length and permission settings
 * \param[in] activate_immediately if true key will be activated immediately
 * \param[in] key_data_enc externally defined encoding of \p key_data
 * \param[in] key_data key secret
 * \param[in] key_data_len length of the secret
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_KID_EXISTS key id is already used
 * \return \ref SDC_KEYLEN_INVALID invalid key length
 * \return else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_import_insecure_product_key(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    const libkeyman_key_spec_t *key_spec,
    bool activate_immediately,
    sdc_keysecret_enc_t key_data_enc,
    const uint8_t *key_data,
    size_t key_data_len);

/**
 * \ingroup libKeyman_API
 * \brief Activate key
 *
 * In case the key is not immediate activated this function can
 * be used to activate a key.
 * This function will be used to prevent e.g. that a persistent key
 * is used before the persistent key-store has been updated.
 * While not active trying to set the key to a session will result in
 * SDC_KEY_NOT_READY
 *
 * \param[in] ctx pointer to libkeyman context
 * \param[in] key_id id of the key
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_KID_NOT_AVAILABLE no such key
 * \return else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_activate_key(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id);

/**
 * \ingroup libKeyman_API
 * \brief Remove a key-storage key
 *
 * \param[in] ctx pointer to libkeyman context
 * \param[in] key_id id of the key
 * \param[in] uid uid of the process requesting remove
 * \param[in] gid gid of the process requesting remove
 * \param[out] result_is_persistent optional pointer to return if the key was a persistent key - required in order to trigger storage of the keystore externally
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_ACCESS_DENIED no permission to remove the key (uid != owning uid)
 * \return \ref SDC_KID_NOT_AVAILABLE no such key
 * \return \ref SDC_KEY_LOCKED key is still used
 * \return else appropriate error value
 */
KEYMAN_API sdc_error_t keyman_remove_storage_key(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    uid_t uid, gid_t gid,
    bool *result_is_persistent
    );

#endif
